#!/usr/bin/env python3
"""
Show whether a running prestack_kids.py process is actively computing.
Prints deltas of user/system CPU time and context switches.

Usage:
  python3 scripts/monitor_proc_activity.py          # lenses.csv run
  python3 scripts/monitor_proc_activity.py randoms  # lenses_random.csv run
"""
import os, sys, time, shutil, subprocess

TARGET = "lenses_random.csv" if (len(sys.argv) > 1 and sys.argv[1].lower().startswith("r")) \
         else "lenses.csv"

def find_pid():
    pids = [p for p in os.popen(f"pgrep -f 'prestack_kids.py .*{TARGET}'").read().split() if p.strip()]
    return int(pids[0]) if pids else None

def read_cpu_stats(pid):
    # /proc/<pid>/stat: fields 14,15 are utime, stime (clock ticks)
    with open(f"/proc/{pid}/stat") as f:
        fields = f.read().split()
        ut, st = int(fields[13]), int(fields[14])
    # /proc/<pid>/status: voluntary/nonvoluntary context switches
    v = nv = 0
    with open(f"/proc/{pid}/status") as f:
        for line in f:
            if line.startswith("voluntary_ctxt_switches"):
                v = int(line.split()[-1])
            elif line.startswith("nonvoluntary_ctxt_switches"):
                nv = int(line.split()[-1])
    return ut, st, v, nv

def main():
    pid = find_pid()
    if pid is None:
        print(f"[INFO] No prestack_kids.py found for {TARGET}.")
        return
    hz = os.sysconf(os.sysconf_names['SC_CLK_TCK'])
    print(f"[OK] Monitoring PID {pid} (target {TARGET}); 1 tick = 1/{hz:.0f}s.")
    ut0, st0, v0, nv0 = read_cpu_stats(pid)
    while True:
        try:
            time.sleep(5)
            ut1, st1, v1, nv1 = read_cpu_stats(pid)
            dut, dst = (ut1 - ut0)/hz, (st1 - st0)/hz
            dv, dnv = (v1 - v0), (nv1 - nv0)
            print(f"ΔCPU_user={dut:6.2f}s  ΔCPU_sys={dst:5.2f}s  "
                  f"Δctx(v/nv)={dv}/{dnv}  (PID {pid})")
            ut0, st0, v0, nv0 = ut1, st1, v1, nv1
        except FileNotFoundError:
            print("[INFO] Process finished.")
            break
        except KeyboardInterrupt:
            break

if __name__ == "__main__":
    main()
